package orchestrator;

import java.util.Set;
import java.util.Vector;

import service.ComposedTransition;
import service.ComposedState;
import service.SimpleTransition;
import service.SimpleState;
import service.State;
import service.TransitionSystem;

public abstract class Orchestrator {
	public Orchestrator(TransitionSystem<SimpleState, SimpleTransition> target,
	          TransitionSystem<SimpleState, SimpleTransition>[] available){}
	
	/**
	 * Method that asks to a given Orchestrator to do an Action using the available services
	 * @param action the action to be done
	 * @return the next Target Service state
	 * @throws Exception Action requested is not available
	 */
	public abstract void doAction(SimpleTransition action) throws Exception;
	
	/**
	 * Method that returns the current Target Service State.
	 * @return the current Target Service State
	 */
	public abstract SimpleState getCurrentTargetState();
	
	/**
	 * Method that returns the current composed state of the available services.
	 * The current state of every available service is contained in this composed state.
	 * @return A composed state representing the state of the available services
	 */
	public abstract ComposedState getCurrentServicesState();
	
	/**
	 * Method that returns the last action executed. If no action has been executed yet,
	 * null is returned.
	 * @return The last executed action.
	 */
	public abstract ComposedTransition getLastExecutedAction();
	
	/**
	 * Method that returns the Composition Transition System.
	 * This transition system is the result of the analysis of the simulation relation
	 * @return The Composition Transition System
	 */
	public abstract TransitionSystem<ComposedState, ComposedTransition> getCompositionTs();
	
	/**
	 * Returns a set of couple of states that are in the simulation relation.
	 * The first State in the array represent a target service state, whereas
	 * the second is a composed state that represent the states of the available services
	 * that can simulate the first state.
	 * @return A set of array of states. Each array has 2 elements.
	 */
	public abstract Set<State[]> getAllSimilarState();
	
	/**
	 * Method that returns a Vector of Arrays of Strings.
	 * Each array is composed of 2 elements, the first contain the name of the state
	 * of the target service, the second contains the name of a composed state result of the simulation
	 * that simulates the selected target service states.
	 * The vector contains alla the relations in the simulation.
	 * @return The vector describing the simulation.
	 */
	public abstract Vector<String[]> getSimulationDataToExport();
	
	/**
	 * Method that returns a Vector of Arrays of Strings.
	 * Each array is composed of 4 elements:
	 * - the first contains the name of the state of the target system
	 * - the second contains the name of the composed state that simulates the state above
	 * - the third is the name of the action that should be done by the target service
	 * - the forth is a string representing the indexes of the available services that can complete the requested action (each index is separated by a "," without spaces like this 1,5,6)
	 * @return The vector describing the composition.
	 */
	public abstract Vector<String[]> getCompositionDataToExport();

	
	/**
	 * Method that returns the transition system representing the asynchronous product of the available
	 * services transitions systems.
	 * @return the transition system representing the asynchronous product.
	 */
	public abstract TransitionSystem<ComposedState, ComposedTransition> getAsyncProduct();
}
